import React, { useEffect, useState } from "react";
import PuzzleBoard from "../components/PuzzleBoard";
import { usePuzzleStore } from "../store/puzzleStore";

export default function PuzzlePage({ level, onFinish }) {
  const { markCompleted, correctCount, wrongCount, totalTime, resetAll } =
    usePuzzleStore();
  const questions = Array.isArray(level?.questions) ? level.questions : [];
  const [qIndex, setQIndex] = useState(0);
  const [finished, setFinished] = useState(false);

  useEffect(() => {
    setQIndex(0);
    setFinished(false);
  }, [level?.level]);

  const handleNext = (wasCorrect, timeTaken) => {
    setQIndex((prev) => {
      const next = prev + 1;
      if (next < questions.length) return next;

      // ✅ All questions completed
      markCompleted?.(level.level);
      setFinished(true);
      if (typeof onFinish === "function") onFinish();
      return prev;
    });
  };

  if (finished) {
    return (
      <div className="flex flex-col items-center justify-center text-center text-white p-8 min-h-[60vh]">
        <h1 className="text-4xl font-bold text-green-400 mb-4">
          🎉 Run Completed!
        </h1>
        <p className="text-lg mb-3">
          You’ve finished all questions for this level.
        </p>
        <div className="bg-white/10 p-4 rounded-xl text-left space-y-2">
          <p>✅ Correct Answers: {correctCount}</p>
          <p>❌ Wrong Answers: {wrongCount}</p>
          <p>⏱️ Total Time: {totalTime}s</p>
        </div>
        <button
          onClick={() => {
            resetAll();
            setFinished(false);
            setQIndex(0);
          }}
          className="mt-6 px-6 py-3 bg-gradient-to-r from-cyan-400 to-violet-500 text-black font-semibold rounded-lg hover:opacity-90 transition-all"
        >
          🔁 Play Again
        </button>
      </div>
    );
  }

  const currentQ = questions[qIndex];

  if (!currentQ)
    return (
      <div className="text-center text-white/70 p-8 min-h-[60vh]">
        ✅ All questions done!
      </div>
    );

  return (
    <div className="flex justify-center p-6">
      <PuzzleBoard key={qIndex} question={currentQ} onNext={handleNext} />
    </div>
  );
}
